#' Plot some analysis of esane data
#' 
#' @param path list of character
#' @param selection character, selection of lines to be analyzed
#' @param sub_path character
#' @return NULL
#' 
#' @export
#' @encoding UTF-8

analyse_esane = function(path, selection, sub_path = "") {

	plot_esane = function(x, main, sub, ylab = NULL, path = NULL) {
		if (!is.null(path)) grDevices::pdf(path, width = 15, height = 8)
		col = grDevices::grey(seq(0.15, .85, length.out = nrow(x)))
		coord = graphics::barplot(x, beside = TRUE, col = col, main = main, sub = sub)
		graphics::segments(
			apply(coord, 2, min) - .5, apply(x, 2, mean, na.rm = TRUE), 
			apply(coord, 2, max) + .5, apply(x, 2, mean, na.rm = TRUE),
			lwd = 2, col = "red")
		graphics::legend("topright", title = "Year", legend = c(row.names(x), "mean"), 
			col = c(col, "red"),
			pch = c(rep(15, nrow(x)), NA), lty = c(rep(0, nrow(x)), 1), lwd = 2, pt.cex = 2)
		if (!is.null(path)) grDevices::dev.off()
	}
	
	prepare_selection = function(x) {
		if (length(selection) > 1) return(t(x))
		result = t(t(x))
		colnames(result) = selection
		return(result)
	}
	
	path_input = file.path(path[["esane"]], "net")
	if (length(dir(path_input)) == 0) {
		warning(sprintf("Nothing available in %s to analyse", path_input))
		return(NULL)
	}
	
	path_output = file.path(path[["analysis"]], "esane", sub_path)
	dir.create(path_output, FALSE, TRUE)
	
	materiel = c("BAC123VALESE", "BAC124VALESE")
	stock = c("BAC152VALESE", "BAC155VALESE", "BAC156VALESE")
	level = "N5"
	files = list.files(path = path_input, pattern = level)
	input = lapply(file.path(path_input, files), utils::read.csv2, row.names = "ACTIVITY")
	names(input) = sapply(strsplit(files, "-"), function(x){x[2]})
	ape = unique(unlist(lapply(input, row.names)))
	input = lapply(input, normalise, ape)
	input = list2array(input)
	
	selection = selection[selection %in% row.names(input)]
	if (length(selection) == 0) return(NULL)
	
	input = input[selection, , , drop = FALSE]
	if (all(is.na(input))) return(NULL)
	
	# Number of employees / activities
	N = prepare_selection(input[, "EMPSALNOMESS", ]) 																								
	plot_esane(N, 
		main = "Number of employee", 
		sub = "EMPSALNOMESS", 
		path = file.path(path_output, "employe_strict.pdf"))
		
	plot_esane(prepare_selection(input[, "ACTENTNOMESS", ]), 
		main = "Number of activity", 
		sub = "ACTENTNOMESS", 
		path = file.path(path_output, "entreprise.pdf"))

	# Number of employees adjusted
	N = N + prepare_selection(input[, "ACTENTNOMESS", ])																					
	plot_esane(N, 
		main = "adjusted number of employees", 
		sub = "EMPSALNOMESS + ACTENTNOMESS", 
		path = file.path(path_output, "employe.pdf"))
	
	plot_esane(N / prepare_selection(input[, "ACTENTNOMESS", ]), 
		main = "Adjusted number of employees per activity", 
		sub = "(EMPSALNOMESS + ACTENTNOMESS) / ACTENTNOMESS", 
		path = file.path(path_output, "employe_entreprise.pdf"))
	
	plot_esane(t(apply(input[, materiel, , drop = FALSE], c(1,3), sum, na.rm = TRUE)) / N, 
		main = "Equipment per adjusted number of employees",
		sub = sprintf("(%s) / (EMPSALNOMESS + ACTENTNOMESS)", paste(materiel, collapse = " + ")),
		ylab = "euros",
		path = file.path(path_output, "materiel.pdf"))

	plot_esane(prepare_selection(input[, "BAC123VALESE", ]) / N, 
		main = "Specific equipment per adjusted number of employees",
		sub = "BAC123VALESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "materiel_specifique.pdf"))
		
	plot_esane(prepare_selection(input[, "BAC124VALESE", ]) / N, 
		main = "Current equipment per adjusted number of employees",
		sub = "BAC124VALESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "materiel_generique.pdf"))

	plot_esane(t(apply(input[, stock, , drop = FALSE], c(1,3), sum, na.rm = TRUE)) / N, 
		main = "Stock excluding prestations by adjusted number of employees",
		sub = sprintf("(%s) / (EMPSALNOMESS + ACTENTNOMESS)", paste(stock, collapse = " + ")),
		ylab = "euros",
		path = file.path(path_output, "stock.pdf"))

	plot_esane(t(apply(input[, stock[-3], , drop = FALSE], c(1,3), sum, na.rm = TRUE)) / N, 
		main = "Stock excluding product and prestations per adjusted number of employees",
		sub = sprintf("(%s) / (EMPSALNOMESS + ACTENTNOMESS)", paste(stock[-3], collapse = " + ")),
		ylab = "euros",
		path = file.path(path_output, "stock_production.pdf"))

	plot_esane(prepare_selection(input[,"BAC152VALESE",]) / N, 
		main = "Raw material per adjusted number of employees",
		sub = "BAC152VALESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "stock_matiere.pdf"))

	plot_esane(prepare_selection(input[,"BAC155VALESE",]) / N, 
		main = "Intermediate products per adjusted number of employees",
		sub = "BAC155VALESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "stock_intermediaire.pdf"))

	plot_esane(prepare_selection(input[,"BAC156VALESE",]) / N, 
		main = "Products per adjusted number of employees",
		sub = "BAC156VALESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "stock_marchandise.pdf"))

	plot_esane(prepare_selection(input[,"BACSTOMPPESE",]) / N, 
		main = "Stock per adjusted number of employees",
		sub = "BACSTOMPPESE / (EMPSALNOMESS + ACTENTNOMESS)", 
		ylab = "euros",
		path = file.path(path_output, "stock_indifferencie.pdf"))
}
