#' @title Adjust absolute H of components
#' 
#' @description
#' `adjust_h_abs()` adjusts the filed `H_abs` in all data.frame of `data_table`.
#' 
#' @param data_table list of data.frame describing components in a model.
#' @param h_abs character defining how the adjustment is done.
#' @param h_ref numeric, of length 1, that gives the H reference in the model.
#' 
#' @return The same list of data.frame, each with fields `H` and `H_abs` that
#' has been
#' adjusted (or maybe created). 
#' 
#' @details
#' 
#' If some of the data.frame in data_table do not have a field `H`, it is added
#' and set to 0. Il some values in any field `H` are missing (NA values), they
#' are set to 0.
#' 
#' `H_abs` is adjusted depending on the value of `h_abs`:
#' 
#' - For **model**, all `H_abs` are adjusted by taking into account the relative
#'   height `H` of the component and the absolute height `H_abs` of its container.
#' - For **room**, all rooms are supposed to be at `H_abs = 0`, then the same
#'   logic as in case **model** is applied. Relative `H` for storeys and rooms
#'   are set to 0 for consistency.
#' - For **component** all relative height `H` and absolute height `H_abs` are
#'   set to 0.
#' 
#' **Note** :
#' - one room is "contained" in one storey
#' - one wall or one furniture is "contained" in one room
#' - one opening or one coating is "contained" in one wall
#' 
#' @keywords internal 

adjust_h_abs = function(data_table, h_abs = c("model", "room", "component"), h_ref = 0) {
    h_abs = match.arg(h_abs)
    h_ref = h_ref[1]

    category = intersect(
        names(data_table),
        c("storey", "room", "wall", "opening", "coating", "furniture")
    )

    # h_abs = component
    # everything is set to O
    if ("component" %in% h_abs) {
        data_table[category] = lapply(
            data_table[category],
            function(x){
                x[["H"]] = x[["H_abs"]] = 0
                return(x)
            }
        )
        return(data_table)
    }

    # h_abs = room
    # storey and room are set to O, then relative H is used for other components
    if ("room" %in% h_abs) {
        data_table[["storey"]]["H"] = 0
        data_table[["room"]]["H"] = 0
        data_table = adjust_h_abs(data_table, "model")
        return(data_table)
    }

    # h_abs = model
    # relative H used for all components

    ## missing relative H set to 0
    data_table[category] = lapply(
        data_table[category],
        function(x){
            if (is.null(x[["H"]])) x[["H"]] = 0
            x[["H"]][is.na(x[["H"]])] = 0
            return(x)
        }
    )

    ## storey adjust from h_ref
    data_table[["storey"]][["H_abs"]] = data_table[["storey"]][["H"]] + h_ref

    ## room adjust from storey
    h_ref = keep_df_order(
        data_table[["room"]][c("storey")],
        merge,
        data_table[["storey"]][c("storey", "H_abs")]
    )
    data_table[["room"]][["H_abs"]] = data_table[["room"]][["H"]] + h_ref[["H_abs"]]

    ## wall adjust from room
    h_ref = keep_df_order(
        data_table[["wall"]][c("room")],
        merge,
        data_table[["room"]][c("room", "H_abs")]
    )
    data_table[["wall"]][["H_abs"]] = data_table[["wall"]][["H"]] + h_ref[["H_abs"]]

    ## opening adjust from wall
    h_ref = keep_df_order(
        data_table[["opening"]][c("id_wall")],
        merge,
        data_table[["wall"]][c("id_wall", "H_abs")]
    )
    data_table[["opening"]][["H_abs"]] = data_table[["opening"]][["H"]] + h_ref[["H_abs"]]

    ## cooating adjust from wall
    if ("coating" %in% category){
        h_ref = keep_df_order(
            data_table[["coating"]][c("id_wall")],
            merge,
            data_table[["wall"]][c("id_wall", "H_abs")]
        )
        data_table[["coating"]][["H_abs"]] = data_table[["coating"]][["H"]] + h_ref[["H_abs"]]
    }

    ## furniture adjust from room
    if ("furniture" %in% category){
        h_ref = keep_df_order(
            data_table[["furniture"]][c("room")],
            merge,
            data_table[["room"]][c("room", "H_abs")]
        )
        data_table[["furniture"]][["H_abs"]] = data_table[["furniture"]][["H"]] + h_ref[["H_abs"]]
    }

    return(data_table)
}